<?php
require_once '../config/database.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!isset($input['email']) || !isset($input['password'])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Email and password are required'
    ]);
    exit;
}

$email = filter_var($input['email'], FILTER_VALIDATE_EMAIL);
$password = $input['password'];

if (!$email) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Invalid email format'
    ]);
    exit;
}

try {
    $db = getDbConnection();
    
    // Hash password
    $passwordHash = hash('sha256', $password);
    
    // Check if user exists and get user data
    $stmt = $db->prepare("
        SELECT id, email, username, profile_picture, user_level, is_verified, is_deleted, password_hash, created_at 
        FROM users 
        WHERE email = ? 
        AND is_deleted = FALSE
    ");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if (!$user) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'Invalid email or password'
        ]);
        exit;
    }
    
    if (!$user['is_verified']) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'Please verify your email first'
        ]);
        exit;
    }
    
    if (!$user['username'] || !$user['password_hash']) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'Please complete your registration first'
        ]);
        exit;
    }
    
    // Verify password
    if ($user['password_hash'] !== $passwordHash) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'Invalid email or password'
        ]);
        exit;
    }
    
    // Remove sensitive data
    unset($user['password_hash']);
    
    // Log successful login (optional)
    $stmt = $db->prepare("
        INSERT INTO user_activity_log (user_id, activity_type, description, ip_address, user_agent) 
        VALUES (?, 'login', 'User logged in successfully', ?, ?)
    ");
    $stmt->execute([
        $user['id'], 
        $_SERVER['REMOTE_ADDR'] ?? null, 
        $_SERVER['HTTP_USER_AGENT'] ?? null
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Login successful',
        'data' => $user
    ]);
    
} catch (Exception $e) {
    error_log("Login error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred during login. Please try again.'
    ]);
}
?>