<?php
require_once '../config/database.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!isset($input['email']) || !isset($input['username']) || !isset($input['password'])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Email, username, and password are required'
    ]);
    exit;
}

$email = filter_var($input['email'], FILTER_VALIDATE_EMAIL);
$username = trim($input['username']);
$password = $input['password'];
$profilePicture = $input['profile_picture'] ?? null;

if (!$email) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Invalid email format'
    ]);
    exit;
}

if (strlen($username) < 3 || strlen($username) > 20) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Username must be between 3 and 20 characters'
    ]);
    exit;
}

if (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Username can only contain letters, numbers, and underscores'
    ]);
    exit;
}

try {
    $db = getDbConnection();
    
    // Check if user exists and is verified
    $stmt = $db->prepare("
        SELECT id, username, is_verified 
        FROM users 
        WHERE email = ? 
        AND is_deleted = FALSE
    ");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if (!$user) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Please verify your email first'
        ]);
        exit;
    }
    
    if (!$user['is_verified']) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Please verify your email first'
        ]);
        exit;
    }
    
    if ($user['username']) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'User already registered'
        ]);
        exit;
    }
    
    // Check if username is already taken
    $stmt = $db->prepare("
        SELECT id FROM users 
        WHERE username = ? 
        AND is_deleted = FALSE
    ");
    $stmt->execute([$username]);
    if ($stmt->fetch()) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Username is already taken'
        ]);
        exit;
    }
    
    // Hash password
    $passwordHash = hash('sha256', $password);
    
    // Set default profile picture if not provided
    if (!$profilePicture) {
        $profilePicture = "https://api.dicebear.com/7.x/fun-emoji/jpg?seed=" . urlencode($email);
    }
    
    // Update user record
    $stmt = $db->prepare("
        UPDATE users 
        SET username = ?, 
            password_hash = ?, 
            profile_picture = ?, 
            user_level = 'registered',
            updated_at = NOW() 
        WHERE id = ?
    ");
    $stmt->execute([$username, $passwordHash, $profilePicture, $user['id']]);
    
    // Get updated user data
    $stmt = $db->prepare("
        SELECT id, email, username, profile_picture, user_level, is_verified, is_deleted, created_at 
        FROM users 
        WHERE id = ?
    ");
    $stmt->execute([$user['id']]);
    $userData = $stmt->fetch();
    
    echo json_encode([
        'success' => true,
        'message' => 'Registration completed successfully',
        'data' => $userData
    ]);
    
} catch (Exception $e) {
    error_log("Registration error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred during registration. Please try again.'
    ]);
}
?>