<?php
require_once '../config/database.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!isset($input['email']) || !isset($input['verification_code']) || !isset($input['new_password'])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Email, verification code, and new password are required'
    ]);
    exit;
}

$email = filter_var($input['email'], FILTER_VALIDATE_EMAIL);
$code = $input['verification_code'];
$newPassword = $input['new_password'];

if (!$email) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Invalid email format'
    ]);
    exit;
}

if (strlen($code) !== 4 || !ctype_digit($code)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Invalid verification code format'
    ]);
    exit;
}

try {
    $db = getDbConnection();
    
    // Check if verification code is valid
    $stmt = $db->prepare("
        SELECT id FROM verification_codes 
        WHERE email = ? 
        AND code = ? 
        AND type = 'password_reset' 
        AND expires_at > NOW() 
        AND used = FALSE
        ORDER BY created_at DESC 
        LIMIT 1
    ");
    $stmt->execute([$email, $code]);
    $verification = $stmt->fetch();
    
    if (!$verification) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Invalid or expired verification code'
        ]);
        exit;
    }
    
    // Check if user exists
    $stmt = $db->prepare("
        SELECT id FROM users 
        WHERE email = ? 
        AND is_deleted = FALSE 
        AND is_verified = TRUE
    ");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if (!$user) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'message' => 'User not found'
        ]);
        exit;
    }
    
    // Start transaction
    $db->beginTransaction();
    
    try {
        // Hash new password
        $passwordHash = hash('sha256', $newPassword);
        
        // Update user password
        $stmt = $db->prepare("
            UPDATE users 
            SET password_hash = ?, updated_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$passwordHash, $user['id']]);
        
        // Mark verification code as used
        $stmt = $db->prepare("
            UPDATE verification_codes 
            SET used = TRUE 
            WHERE id = ?
        ");
        $stmt->execute([$verification['id']]);
        
        // Log password reset activity
        $stmt = $db->prepare("
            INSERT INTO user_activity_log (user_id, activity_type, description, ip_address, user_agent) 
            VALUES (?, 'password_reset', 'Password reset successfully', ?, ?)
        ");
        $stmt->execute([
            $user['id'], 
            $_SERVER['REMOTE_ADDR'] ?? null, 
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
        
        $db->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Password reset successfully'
        ]);
        
    } catch (Exception $e) {
        $db->rollback();
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("Password reset error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred during password reset. Please try again.'
    ]);
}
?>