<?php
require_once '../config/database.php';
require_once '../config/email_service.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!isset($input['email']) || empty($input['email'])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Email is required'
    ]);
    exit;
}

$email = filter_var($input['email'], FILTER_VALIDATE_EMAIL);
if (!$email) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Invalid email format'
    ]);
    exit;
}

try {
    $db = getDbConnection();
    
    // Check rate limiting - max 5 attempts per hour
    $stmt = $db->prepare("
        SELECT COUNT(*) as attempt_count 
        FROM verification_codes 
        WHERE email = ? 
        AND type = 'email_verification' 
        AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)
    ");
    $stmt->execute([$email]);
    $result = $stmt->fetch();
    
    if ($result['attempt_count'] >= 5) {
        http_response_code(429);
        echo json_encode([
            'success' => false,
            'message' => 'Too many verification attempts. Please try again later.'
        ]);
        exit;
    }
    
    // Generate 4-digit verification code
    $code = str_pad(random_int(0, 9999), 4, '0', STR_PAD_LEFT);
    
    // Store verification code in database
    $stmt = $db->prepare("
        INSERT INTO verification_codes (email, code, type, expires_at) 
        VALUES (?, ?, 'email_verification', DATE_ADD(NOW(), INTERVAL 10 MINUTE))
    ");
    $stmt->execute([$email, $code]);
    
    // Send email
    $emailSent = sendVerificationEmail($email, $code);
    
    if ($emailSent) {
        echo json_encode([
            'success' => true,
            'message' => 'Verification code sent successfully'
        ]);
    } else {
        // Remove the code from database if email failed
        $stmt = $db->prepare("DELETE FROM verification_codes WHERE email = ? AND code = ?");
        $stmt->execute([$email, $code]);
        
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Failed to send verification email. Please try again.'
        ]);
    }
    
} catch (Exception $e) {
    error_log("Send verification error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred. Please try again.'
    ]);
}
?>