<?php
require_once '../config/database.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!isset($input['email']) || !isset($input['verification_code'])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Email and verification code are required'
    ]);
    exit;
}

$email = filter_var($input['email'], FILTER_VALIDATE_EMAIL);
$code = $input['verification_code'];

if (!$email) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Invalid email format'
    ]);
    exit;
}

if (strlen($code) !== 4 || !ctype_digit($code)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Invalid verification code format'
    ]);
    exit;
}

try {
    $db = getDbConnection();
    
    // Check if verification code is valid
    $stmt = $db->prepare("
        SELECT id FROM verification_codes 
        WHERE email = ? 
        AND code = ? 
        AND type = 'email_verification' 
        AND expires_at > NOW() 
        AND used = FALSE
        ORDER BY created_at DESC 
        LIMIT 1
    ");
    $stmt->execute([$email, $code]);
    $verification = $stmt->fetch();
    
    if (!$verification) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Invalid or expired verification code'
        ]);
        exit;
    }
    
    // Start transaction
    $db->beginTransaction();
    
    try {
        // Mark verification code as used
        $stmt = $db->prepare("
            UPDATE verification_codes 
            SET used = TRUE 
            WHERE id = ?
        ");
        $stmt->execute([$verification['id']]);
        
        // Check if user already exists
        $stmt = $db->prepare("
            SELECT id, is_verified 
            FROM users 
            WHERE email = ? 
            AND is_deleted = FALSE
        ");
        $stmt->execute([$email]);
        $user = $stmt->fetch();
        
        if ($user) {
            // Update existing user verification status
            $stmt = $db->prepare("
                UPDATE users 
                SET is_verified = TRUE, updated_at = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$user['id']]);
        } else {
            // Create new user record (will be completed during registration)
            $stmt = $db->prepare("
                INSERT INTO users (email, is_verified, created_at) 
                VALUES (?, TRUE, NOW())
            ");
            $stmt->execute([$email]);
        }
        
        $db->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Email verified successfully'
        ]);
        
    } catch (Exception $e) {
        $db->rollback();
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("Email verification error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred during verification. Please try again.'
    ]);
}
?>